export class WebRTCPlayer {
	constructor() {
		const PeerConnection = window.RTCPeerConnection || window.mozRTCPeerConnection || window.webkitRTCPeerConnection;
		if (!PeerConnection) {
			throw "系统不支持";
		}
		try {
			this.pc = new RTCPeerConnection();
			this.pc.onaddstream = (event) => {
				this.stream = event.stream;
				if (this.onPlay) {
					this.onPlay()
				}
			};
			this.pc.onconnectionstatechange  =(event) => {
				console.log('webrtc state', Date.now(), event.target.connectionState)
				if (this.onStatus) {
					this.onStatus(event.target.connectionState)
				}
			}
			// this.pc.oniceconnectionstatechange =(event) => {
			// 	console.log("webrtc ice state", Date.now(), event.target.iceConnectionState)
			// 	if (event.target.iceConnectionState === 'disconnected') {
			// 		if (this.onStatus) {
			// 			this.onStatus(event.target.iceConnectionState)
			// 		}
			// 	}
			// }
		} catch(e) {
			throw "系统不支持";
		}	
	}

	request(opts) {
		return new Promise((resolve, reject) => {
			opts = opts || {};
			opts = Object.assign({}, {
				type: "POST",
				dataType: "json",
				async: true // 异步请求
			}, opts);
			opts.data = opts.data || {};
			let xhr = {};
			if (window.XMLHttpRequest) {
				// W3C标准
				xhr = new XMLHttpRequest();
			} else {
				// IE标准
				xhr = new ActiveXObject("Microsoft.XMLHTTP");
			}
			xhr.onreadystatechange = () => {
				if (xhr.readyState == 4) {
					const status = xhr.status;
					if (status >= 200 && status < 300) {
						resolve(JSON.parse(xhr.responseText));
					} else {
						reject('网络异常');
					}
				}
			};
			if (opts.type == "GET") {
				const query = [];
				for (const key in opts.data) {
					if (typeof opts.data[key] === "object") {
						opts.data[key] = JSON.stringify(opts.data[key])
					}
					query.push(encodeURIComponent(key) + "=" + encodeURIComponent(opts.data[key]));
				}
				// 对需要传入的参数的处理
				const params = query.join("&");
				xhr.open("GET", opts.url + "?" + params, opts.async);
				xhr.send(null);
			} else if (opts.type == "POST") {
				//打开请求
				xhr.open("POST", opts.url, opts.async);
				//POST请求设置请求头
				if (opts.dataType === 'json') {
					xhr.setRequestHeader("Content-Type", "application/json");
				}
				//发送请求参数
				xhr.send(JSON.stringify(opts.data));
			}
		});
	}
	
	async play(url) {
		this.pc.addTransceiver("audio", {direction: "recvonly"});
		this.pc.addTransceiver("video", {direction: "recvonly"});
		const offer = await this.pc.createOffer({
			offerToReceiveAudio: 1,
			offerToReceiveVideo: 1
		});
		await this.pc.setLocalDescription(offer);
		if (/^webrtc:/ig.test(url)) {
			const conf = this.prepareUrl(url);
			const apiUrl = conf.apiUrl;
			const streamUrl = conf.streamUrl;
			const data = {
				api: apiUrl,
				streamurl: streamUrl,
				clientip: null,
				sdp: offer.sdp
			};
			const res = await this.request({
				url: apiUrl,
				data: data,
			});
			if (res.code) {
				throw '网络异常';
			}
			console.log('webrtc sdp', res.sdp)
			this.pc.setRemoteDescription(
				new RTCSessionDescription({type: 'answer', sdp: res.sdp})
			);
		}
	}

	close() {
		this.pc.close();
	}

	// 格式化trmp url
	parseRtmpUrl(rtmpUrl) {
		const a = document.createElement("a");
		a.href = rtmpUrl.replace("rtmp://", "http://").replace("webrtc://", "http://").replace("rtc://", "http://");
		const vhost = a.hostname;
		let app = a.pathname.substr(1, a.pathname.lastIndexOf("/") - 1);
		const stream = a.pathname.substr(a.pathname.lastIndexOf("/") + 1);

		// parse the vhost in the params of app, that srs supports.
		app = app.replace("...vhost...", "?vhost=");
		if (app.indexOf("?") >= 0) {
			const params = app.substr(app.indexOf("?"));
			app = app.substr(0, app.indexOf("?"));

			if (params.indexOf("vhost=") > 0) {
				vhost = params.substr(params.indexOf("vhost=") + "vhost=".length);
				if (vhost.indexOf("&") > 0) {
					vhost = vhost.substr(0, vhost.indexOf("&"));
				}
			}
		}

		// when vhost equals to server, and server is ip,
		// the vhost is __defaultVhost__
		if (a.hostname === vhost) {
			const re = /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/;
			if (re.test(a.hostname)) {
				vhost = "__defaultVhost__";
			}
		}
		
		// parse the schema
		let schema = "rtmp";
		if (rtmpUrl.indexOf("://") > 0) {
			schema = rtmpUrl.substr(0, rtmpUrl.indexOf("://"));
		}

		let port = a.port;
		if (!port) {
			if (schema === 'http') {
				port = 80;
			} else if (schema === 'https') {
				port = 443;
			} else if (schema === 'rtmp') {
				port = 1935;
			}
		}

		const ret = {
			url: rtmpUrl,
			schema: schema,
			server: a.hostname, 
            port: port,
			vhost: vhost, app: app, stream: stream
		};
		let queryStr = a.search;
		ret.user_query = {};
		if (queryStr.length > 0) {
			// split again for angularjs.
			if (queryStr.indexOf("?") >= 0) {
				queryStr = queryStr.split("?")[1];
			}
			const queries = queryStr.split("&");
			for (const i = 0; i < queries.length; i++) {
				const elem = queries[i];
				const query = elem.split("=");
				ret[query[0]] = query[1];
				ret.user_query[query[0]] = query[1];
			}
			// alias domain for vhost.
			if (ret.domain) {
				ret.vhost = ret.domain;
			}
		}

		// For webrtc API, we use 443 if page is https, or schema specified it.
		if (!ret.port) {
			if (schema === 'webrtc' || schema === 'rtc') {
				if (ret.user_query.schema === 'https') {
					ret.port = 443;
				} else if (window.location.href.indexOf('https://') === 0) {
					ret.port = 443;
				} else {
					// For WebRTC, SRS use 1985 as default API port.
					ret.port = 1985;
				}
			}
		}

		return ret;
	};

	// 生成RTC API url
	prepareUrl(url) {
		let apiUrl, streamUrl;
		const urlObject = this.parseRtmpUrl(url);
		let schema = urlObject.user_query.schema;
		schema = schema? schema + ':' : window.location.protocol;
		let port = urlObject.port || 1985;
		if (schema === 'https:') {
			port = urlObject.port || 443;
		}
		if (schema === 'file:') {
			schema = 'http:';
		}
		let api = urlObject.user_query.play || '/rtc/v1/play/';
		if (api.lastIndexOf('/') !== api.length - 1) {
			api += '/';
		}
		apiUrl = schema + '//' + urlObject.server + /* ':' + port +*/ api;
		for (const key in urlObject.user_query) {
			if (key !== 'api' && key !== 'play') {
				apiUrl += '&' + key + '=' + urlObject.user_query[key];
			}
		}
		// Replace /rtc/v1/play/&k=v to /rtc/v1/play/?k=v
		apiUrl = apiUrl.replace(api + '&', api + '?');
		streamUrl = urlObject.url;
		return {apiUrl: apiUrl, streamUrl: streamUrl, schema: schema, urlObject: urlObject, port: port};
	}
}
